/////////////////////////////////////////////////////////////////////////////////

// Original obtained from GlsSandbox.com
// Adapted, initialy trivialy, for VGHD by TheEmu.
//
// Modified for EverthangForever so that the selection of which three  functions
// are used can be specified via a uniform parameter in the VGHD scene file.  At
// some point I am very likely to implement a shader of my own based on this one
// but with much more cutomisation possible via the scene file, but all that has
// been done is to use uniforms for function selection and the maximum iteration
// count together with some minor rearrangement of part of the code.  If I do go
// any further with this I will probably make its interface similar to that used
// for my Tunnel A shader and name it accordingly. TheEmu.
//
/////////////////////////////////////////////////////////////////////////////////

// The function "trap" calculates min(max(A,B),C) where A, B and C are  obtained
// by evaluating functions of the current pixel's position.  Which functions are
// specified by the funcA, funcB and funcC uniform parameters. Each of the these
// may be either a function id or a negative function id. Function ids are small
// positive integers and select a function from the following list:- 

   #define func1(p) abs(max(abs(p.z)-0.1,abs(p.x)-0.1))-0.01
   #define func2(p) length(max(abs(p.xy)-0.05,0.0))
   #define func3(p) length(p)-0.5
   #define func4(p) length(max(abs(p)-0.35,0.0))
   #define func5(p) abs(length(p.xz)-0.2)-0.01
   #define func6(p) abs(min(torus(vec3(p.x,mod(p.y,0.4)-0.2,p.z), vec2(0.1,0.05)), max(abs(p.z)-0.05,abs(p.x)-0.05)))-0.005
   #define func7(p) abs(min(torus(p,vec2(0.3,0.05)), max(abs(p.z)-0.05, abs(p.x)-0.05)))-0.005
   #define func8(p) min(length(p.xz), min(length(p.yz), length(p.xy))) - 0.05
 
// If the function id is negated then the value of the function is negated  and  a
// function id of zero always results in 0.0. So, for example, funcSelect = 7,-5,2
// will evaluate min(max(func7,-func5),func2).  Invalid  function  ids are treated
// as if they are 0.

uniform int funcA;
uniform int funcB;
uniform int funcC;

// The maximum iteration count can be set in the scn file. If not specified, or is
// explicitly specified as 0, then a defualt value of 70 is used.

uniform int maxIter;

   int MAX_ITER = (maxIter==0) ? 70 : maxIter;

// If DataPanel is any non zero value then a data panel showing the values of some
// shader control parameters is dislayed. See the body of the shader for details.

uniform int DataPanel;

/////////////////////////////////////////////////////////////////////////////////

// Standard shader inputs

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define time u_Elapsed
#define resolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

#ifdef GL_ES
precision highp float;
#endif

vec3 iResolution = vec3(resolution,1.0);

vec2 rotate(in vec2 v, in float a) {
   return vec2(cos(a)*v.x + sin(a)*v.y, -sin(a)*v.x + cos(a)*v.y);
}

float torus(in vec3 p, in vec2 t)
{
   vec2 q = abs(vec2(max(abs(p.x), abs(p.z))-t.x, p.y));
   return max(q.x, q.y)-t.y;
}

float trap(in vec3 p)
{
   float A = 0.0;
   float B = 0.0;
   float C = 0.0;

   int signA = sign(funcA);
   int signB = sign(funcB);
   int signC = sign(funcC);

   switch ( funcA*signA )
    { case 1 : A = func1(p); break;
      case 2 : A = func2(p); break;
      case 3 : A = func3(p); break;
      case 4 : A = func4(p); break;
      case 5 : A = func5(p); break;
      case 6 : A = func6(p); break;
      case 7 : A = func7(p); break;
      case 8 : A = func8(p); break;
    }

   switch ( funcB*signB )
    { case 1 : B = func1(p); break;
      case 2 : B = func2(p); break;
      case 3 : B = func3(p); break;
      case 4 : B = func4(p); break;
      case 5 : B = func5(p); break;
      case 6 : B = func6(p); break;
      case 7 : B = func7(p); break;
      case 8 : B = func8(p); break;
    }

   switch ( funcC*signC )
    { case 1 : C = func1(p); break;
      case 2 : C = func2(p); break;
      case 3 : C = func3(p); break;
      case 4 : C = func4(p); break;
      case 5 : C = func5(p); break;
      case 6 : C = func6(p); break;
      case 7 : C = func7(p); break;
      case 8 : C = func8(p); break;
    }

   A = A * signA;
   B = B * signB;
   C = C * signC;

   return min ( max(A,B), C );

}

float map(in vec3 p)
{
   float cutout = dot(abs(p.yz),vec2(0.5))-0.035;
   float road = max(abs(p.y-0.025), abs(p.z)-0.035);

   vec3 z = abs(1.0-mod(p,2.0));
   z.yz = rotate(z.yz, time*0.05);

   float d = 999.0;
   float s = 1.0;
   for (float i = 0.0; i < 3.0; i++) {
      z.xz = rotate(z.xz, radians(i*10.0+time));
      z.zy = rotate(z.yz, radians((i+1.0)*20.0+time*1.1234));
      z = abs(1.0-mod(z+i/3.0,2.0));

      z = z*2.0 - 0.3;
      s *= 0.5;
      d = min(d, trap(z) * s);
   }
   return min(max(d, -cutout), road);
}

vec3 hsv(in float h, in float s, in float v) {
   return mix(vec3(1.0), clamp((abs(fract(h + vec3(3, 2, 1) / 3.0) * 6.0 - 3.0) - 1.0), 0.0 , 1.0), s) * v;
}

vec3 intersect(in vec3 rayOrigin, in vec3 rayDir)
{
   float total_dist = 0.0;
   vec3 p = rayOrigin;
   float d = 1.0;
   float iter = 0.0;
   float mind = 3.14159+sin(time*0.1)*0.2;

   for (int i = 0; i < MAX_ITER; i++)
   {
      if (d < 0.001) continue;

      d = map(p);
      p += d*vec3(rayDir.x, rotate(rayDir.yz, sin(mind)));
      mind = min(mind, d);
      total_dist += d;
      iter++;
   }

   vec3 color = vec3(0.0);
   if (d < 0.001) {
      float x = (iter/float(MAX_ITER));
      float y = (d-0.01)/0.01/(float(MAX_ITER));
      float z = (0.01-d)/0.01/float(MAX_ITER);
      if (max(abs(p.y-0.025), abs(p.z)-0.035)<0.002) { // Road
         float w = smoothstep(mod(p.x*50.0, 4.0), 2.0, 2.01);
         w -= 1.0-smoothstep(mod(p.x*50.0+2.0, 4.0), 2.0, 1.99);
         w = fract(w+0.0001);
         float a = fract(smoothstep(abs(p.z), 0.0025, 0.0026));
        color = vec3((1.0-x-y*2.)*mix(vec3(0.8, 0.1, 0), vec3(0.1), 1.0-(1.0-w)*(1.0-a)));

      } else {
         float q = 1.0-x-y*2.+z;
         color = hsv(q*0.2+0.85, 1.0-q*0.2, q);
      }
   } else
      // was color = hsv(d, 1.0, 1.0)*mind*45.0; // Background
	  color = hsv(d, 0.3, 1.0)*mind*45.0; // Background
   return color;
}

//---------------------------------------------

// A very crude way of displaying numerical values using a very ugly font. The
// image for each of the digits 0 to 9 an array of 5 lines of 3 dots  which is
// encoded as a 15 bit integer. This is expanded to a larger character size by
// simply scaling each of the encoded dots to give larger square dots  in  the
// final image. Each character is positioned at the center of a slighly larger
// cell so that the characters do not touch each other. The result is not very
// pretty (especially the 7), but is not too bad and is useful for debugging.

const vec2 char_size = vec2(12.0,20.0); // 3x5 character scaled by 4
const vec2 char_cell = vec2(16.0,24.0); // 4x6 cell scaled up by 4

const vec2 char_cell_border = ( char_size - char_cell ) * 0.5;

// The colours to use as the characters' background and foreground. This ought
// to be a const, but constant arrays are not supported by the language.

vec3[2] char_colour = vec3[2](vec3(0.0),vec3(1.0));

// Each character requires only 15 bits to encode it so they can be encoded as
// 16 bit integers should that be all that is available.  The encoding is done
// by using 5 octal digits, one per line of the character, each of 3 bits. The
// GLSL shader language uses the same horrible convention as C for octal, i.e.
// a leading 0 means that the number is in octal, yeuk,  but so be it.  Again,
// this data ought to be declared as constant,  but  constant  arrays  are not
// supported by the language.

int digit_chars[10]
 = int[10](075557,013111,071747,071717,055711,
           074717,074757,071244,075757,075711);

// A few extra characters are also defined.

int other_chars[16] 
 = int[16] ( 000000, //  0 - Blank
             000700, //  1 - Minus sign
             002720, //  2 - Plus sign
             000200, //  3 - Dot
             000500, //  4 - Dot Dot
             000600, //  5 - Short dash
             002020, //  6 - Colon
             022222, //  7 - Vertical bar 1
             022022, //  8 - Vertical bar 2
             020202, //  9 - Vertical bar 3
             002220, // 10 - Vertical bar 4
             022202, // 11 - Exclamation
             001240, // 12 - Slash
             004210, // 13 - Backslash
             005250, // 14 - Multiply
             003030  // 15 - Equals sign
           );

// Output individual characters.

#define put_blank(p)  put_char ( p, other_chars[0] );
#define put_minus(p)  put_char ( p, other_chars[1] );
#define put_plus(p)   put_char ( p, other_chars[2] );
#define put_dot(p)    put_char ( p, other_chars[3] );
#define put_dotdot(p) put_char ( p, other_chars[4] );
#define put_dash(p)   put_char ( p, other_chars[5] );
#define put_colon(p)  put_char ( p, other_chars[6] );
#define put_vline1(p) put_char ( p, other_chars[7] );
#define put_vline2(p) put_char ( p, other_chars[8] );
#define put_vline3(p) put_char ( p, other_chars[9] );
#define put_vline4(p) put_char ( p, other_chars[10] );
#define put_exclam(p) put_char ( p, other_chars[11] );
#define put_slash(p)  put_char ( p, other_chars[12] );
#define put_bslash(p) put_char ( p, other_chars[13] );
#define put_times(p)  put_char ( p, other_chars[14] );
#define put_equals(p) put_char ( p, other_chars[15] );

// Skip some multiple or fraction of a character cell.

#define put_gap(p,f) p -= vec2(char_cell.x*f,0.0);

// put_char - outputs a single character (e.g. a digit or a sign)

void put_char ( inout vec2 p, int c )
 {
   p -= vec2(char_cell.x,0.0);

   if ( any ( lessThan    ( gl_FragCoord.xy, p             ) ) ) return;
   if ( any ( greaterThan ( gl_FragCoord.xy, p + char_cell ) ) ) return;

   vec2 q = ( gl_FragCoord.xy + char_cell_border - p ) / char_size;

   int k = 0;

   if ( q == fract(q) )
    {
     k = int(floor(5.0*q.y))*3 - int(floor(3.0*q.x)) + 2;
     k = ( c >> k ) & 1;
    }

   gl_FragColor = vec4 ( char_colour[k], 1.0 );

 }

// put_digit - outputs a single digit

void put_digit ( inout vec2 p, int d )
 { 
   put_char ( p, digit_chars[d-(d/10)*10] );
 }

// put sign = outputs the sign of d.  The third argument, s,
// specifies the style to be used for non-negative values.
//
//    s = 0 - no output when d is not negative
//    s = 1 - use blank when d is not negative
//    s = 2 - plus sign when d is not negative
//
// a minus sign will always be output for negative values. 

void put_sign ( inout vec2 p, int d, int s )
 {
   if ( d<0 ) 
    { 
      put_minus ( p );
    }
   else
    {
      switch ( s )
       { case 0 : break;
         case 1 : put_blank ( p ); break;
         case 2 : put_plus  ( p ); break;
       }
    }
 }

void put_int ( inout vec2 p, int v, int s  )
 {
   int x = abs(v);
   do 
    { put_digit ( p, x );
      x /= 10;
    } while ( x > 0 );
    put_sign ( p, v, s );
 }

// -----------------------------------------------------------------------------


void main ( void )
 {
   vec3 upDirection = vec3(0, -1, 0);
   vec3 cameraDir = vec3(1,0,0);
   vec3 cameraOrigin = vec3(time*0.551, 0, 0);

   vec3 u = normalize(cross(upDirection, cameraOrigin));
   vec3 v = normalize(cross(cameraDir, u));
   vec2 screenPos = -1.0 + 2.0 * gl_FragCoord.xy / iResolution.xy;
   screenPos.x *= iResolution.x / iResolution.y;
   vec3 rayDir = normalize(u * screenPos.x + v * screenPos.y + cameraDir*(1.0-length(screenPos)*0.5));

   gl_FragColor = vec4(intersect(cameraOrigin, rayDir), 1.0);

   // If a data pael has been requested then generate one in the
   // bottom right of the window. The way in which the panel put
   // functions work means that the panel items have to be "put"
   // in reverse order,  so the code below will generate a panel
   // showing the three selected functions A, B, C in that order
   // followed by the maximum iteration count. 

   if ( DataPanel != 0 )
       {
         vec2 pp = vec2 ( u_WindowSize.x, 0.0 );
   
         put_int  ( pp, MAX_ITER, 0 ); put_vline3 ( pp );
         put_int  ( pp, funcC, 0 );    put_vline3 ( pp );
         put_int  ( pp, funcB, 0 );    put_vline3 ( pp );
         put_int  ( pp, funcA, 0 ); 

   }

}